/*
 * watchdog.c - driver for Freescale IMX53 on-chip watchdog
 */

#include <common.h>
#include <watchdog.h>
#include <asm-arm/arch-mx53/mx53.h>

#define WDOG_WCR  0x00
#define WDOG_WSR  0x02
#define WDOG_WRSR 0x04
#define WDOG_WICR 0x06
#define WDOG_WMCR 0x08
#define WCR_WDW_BIT   0x0080 /* Disables during Low power WAIT */
#define WCR_WDA_BIT   0x0020 /* N_WDOG out assertion */
#define WCR_SRS_BIT   0x0010 /* Send software reset signal N_WDOG_RST */
#define WCR_WDT_BIT   0x0008 /* N_WDOG1_B out timeout assertion */
#define WCR_WDE_BIT	  0x0004 /* Watchdog enable */
#define WCR_WDBG_BIT  0x0002 /* Suspend watchdog in DEBUG mode */
#define WCR_WDZST_BIT 0x0001 /* Suspend watchdog in STOP and DOZE mode */

#define WCR_WRSR_POR_BIT  0x0010 /* Reboot reason: Power-on-reset */
#define WCR_WRSR_TOUT_BIT 0x0002 /* Reboot reason: time-out */
#define WCR_WRSR_SFTW_BIT 0x0001 /* Reboot reason: software reset */

#define WDOG_DEFAULT_TIMEOUT 120

#ifdef CONFIG_HW_WATCHDOG
void hw_watchdog_reset(void)
{
	/* Reset timer. Pet the watchdog. */
	__REG16(WDOG1_BASE_ADDR + WDOG_WSR) = 0x5555;
	__REG16(WDOG1_BASE_ADDR + WDOG_WSR) = 0xAAAA;
}

void hw_watchdog_init(void)
{
	u16 val;
	const char *wdog_boot_reason = "unknown";
	val = __REG16(WDOG1_BASE_ADDR + WDOG_WRSR);
	if(val & WCR_WRSR_TOUT_BIT) {
		wdog_boot_reason = "WDOG";
	}
	else if(val & WCR_WRSR_SFTW_BIT) {
		wdog_boot_reason = "RST";
	}
	else if(val & WCR_WRSR_POR_BIT) {
		wdog_boot_reason = "POR";
	}
	printf("Watchdog Reboot Reason [0x%04x] = [%s]\n", val, wdog_boot_reason);

	/* Cancel the power-down timer. */
	__REG16(WDOG1_BASE_ADDR + WDOG_WMCR) = 0;

	/* Configure */
	/* WDA and SRS are enabled by default, leave those set. */
	val = __REG16(WDOG1_BASE_ADDR + WDOG_WCR);
	val |= ((WDOG_DEFAULT_TIMEOUT * 2) << 8); /* Set timeout to be 120 seconds. */
	val |= WCR_WDBG_BIT | WCR_WDZST_BIT;      /* Suspend wdt in DEBUG and low-power modes */
	val |= WCR_WDE_BIT;	                      /* Enable the watchdog timer */
	__REG16(WDOG1_BASE_ADDR + WDOG_WCR) = val;
}
#endif
